<?php

namespace App\Http\Controllers;

use App\Models\Bot;
use Illuminate\Http\Request;
use ZipArchive;

class BotController extends Controller
{
    public function index()
    {
        $bots = Bot::latest()->get();
        return view('bots.index', compact('bots'));
    }

    public function create()
    {
        return view('bots.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'command' => 'required|string',
            'bot_file' => 'required|file|mimes:zip|max:51200' // 50MB
        ]);

        try {
            $file = $request->file('bot_file');
            $fileName = time() . '_' . preg_replace('/[^A-Za-z0-9_\-]/', '_', pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME));
            $botPath = storage_path('app/bots/' . $fileName);

            // Simpan file zip
            $zipPath = storage_path('app/bots/' . $fileName . '.zip');

            // Buat folder bots jika belum ada
            if (!file_exists(storage_path('app/bots'))) {
                mkdir(storage_path('app/bots'), 0755, true);
            }

            $file->move(storage_path('app/bots/'), $fileName . '.zip');

            // Ekstrak zip
            $zip = new ZipArchive;
            if ($zip->open($zipPath) === TRUE) {
                if (!file_exists($botPath)) {
                    mkdir($botPath, 0755, true);
                }
                $zip->extractTo($botPath);
                $zip->close();
                unlink($zipPath); // Hapus file zip setelah ekstrak
            } else {
                return back()->with('error', 'Gagal ekstrak file ZIP!');
            }

            // Simpan ke database
            Bot::create([
                'name' => $request->name,
                'command' => $request->command,
                'file_path' => $botPath,
                'status' => 'stopped'
            ]);

            return redirect()->route('bots.index')->with('success', 'Bot berhasil ditambahkan!');

        } catch (\Exception $e) {
            return back()->with('error', 'Gagal menambahkan bot: ' . $e->getMessage());
        }
    }

    public function start(Bot $bot)
    {
        if ($bot->status === 'running') {
            return back()->with('error', 'Bot sudah berjalan!');
        }

        try {
            $isWindows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';

            if ($isWindows) {
                // Windows
                $logFile = $bot->file_path . '/output.log';
                $escapedPath = escapeshellarg($bot->file_path);
                $command = "cd /d $escapedPath && start /B {$bot->command} > output.log 2>&1";
                pclose(popen($command, 'r'));
                $pid = null;
            } else {
                // Linux/Unix
                $logFile = $bot->file_path . '/output.log';
                $command = sprintf(
                    'cd %s && nohup %s > output.log 2>&1 & echo $!',
                    escapeshellarg($bot->file_path),
                    $bot->command
                );

                $output = shell_exec($command);
                $pid = trim($output);

                // Validasi PID harus angka
                if (!preg_match('/^\d+$/', $pid)) {
                    $pid = null;
                }
            }

            $bot->update([
                'status' => 'running',
                'pid' => $pid,
                'log' => null // Reset log
            ]);

            $message = 'Bot berhasil dijalankan!';
            if ($pid) {
                $message .= ' (PID: ' . $pid . ')';
            }

            return back()->with('success', $message);

        } catch (\Exception $e) {
            $bot->update(['log' => $e->getMessage()]);
            return back()->with('error', 'Gagal menjalankan bot: ' . $e->getMessage());
        }
    }

    public function restart(Bot $bot)
    {
        try {
            // Stop bot dulu
            if ($bot->status === 'running' && $bot->pid) {
                $isWindows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';

                if ($isWindows) {
                    shell_exec("taskkill /PID {$bot->pid} /F 2>&1");
                } else {
                    shell_exec("kill -9 {$bot->pid} 2>&1");
                }

                // Tunggu sebentar
                sleep(2);
            }

            // Start lagi
            $isWindows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';

            if ($isWindows) {
                $logFile = $bot->file_path . '/output.log';
                $escapedPath = escapeshellarg($bot->file_path);
                $command = "cd /d $escapedPath && start /B {$bot->command} > output.log 2>&1";
                pclose(popen($command, 'r'));
                $pid = null;
            } else {
                $logFile = $bot->file_path . '/output.log';
                $command = sprintf(
                    'cd %s && nohup %s > output.log 2>&1 & echo $!',
                    escapeshellarg($bot->file_path),
                    $bot->command
                );

                $output = shell_exec($command);
                $pid = trim($output);

                if (!preg_match('/^\d+$/', $pid)) {
                    $pid = null;
                }
            }

            $bot->update([
                'status' => 'running',
                'pid' => $pid,
                'log' => null
            ]);

            return back()->with('success', 'Bot berhasil direstart!');

        } catch (\Exception $e) {
            return back()->with('error', 'Gagal restart bot: ' . $e->getMessage());
        }
    }

    public function logs(Bot $bot)
    {
        $logFile = $bot->file_path . '/output.log';

        if (file_exists($logFile)) {
            $logs = file_get_contents($logFile);
        } else {
            $logs = 'Log file tidak ditemukan.';
        }

        return view('bots.logs', compact('bot', 'logs'));
    }

    public function clearLogs(Bot $bot)
    {
        $logFile = $bot->file_path . '/output.log';

        if (file_exists($logFile)) {
            file_put_contents($logFile, '');
        }

        $bot->update(['log' => null]);

        return redirect()->route('bots.logs', $bot)->with('success', 'Log berhasil dibersihkan!');
    }

    public function stop(Bot $bot)
    {
        if ($bot->status === 'stopped') {
            return back()->with('error', 'Bot sudah berhenti!');
        }

        try {
            if ($bot->pid) {
                $isWindows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';

                if ($isWindows) {
                    shell_exec("taskkill /PID {$bot->pid} /F 2>&1");
                } else {
                    shell_exec("kill -9 {$bot->pid} 2>&1");
                }
            }

            $bot->update([
                'status' => 'stopped',
                'pid' => null
            ]);

            return back()->with('success', 'Bot berhasil dihentikan!');

        } catch (\Exception $e) {
            return back()->with('error', 'Gagal menghentikan bot: ' . $e->getMessage());
        }
    }

    public function destroy(Bot $bot)
    {
        try {
            // Hentikan bot jika masih berjalan
            if ($bot->status === 'running' && $bot->pid) {
                $isWindows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';

                if ($isWindows) {
                    shell_exec("taskkill /PID {$bot->pid} /F 2>&1");
                } else {
                    shell_exec("kill -9 {$bot->pid} 2>&1");
                }
            }

            // Hapus folder bot
            if (is_dir($bot->file_path)) {
                $this->deleteDirectory($bot->file_path);
            }

            $bot->delete();

            return redirect()->route('bots.index')->with('success', 'Bot berhasil dihapus!');

        } catch (\Exception $e) {
            return back()->with('error', 'Gagal menghapus bot: ' . $e->getMessage());
        }
    }

    private function deleteDirectory($dir)
    {
        if (!file_exists($dir)) {
            return true;
        }

        if (!is_dir($dir)) {
            return unlink($dir);
        }

        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }

            if (!$this->deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
                return false;
            }
        }

        return rmdir($dir);
    }

    // Optional: API untuk check status real-time
    public function status()
    {
        try {
            $bots = Bot::all(['id', 'status', 'pid']);

            // Check if process still running
            foreach ($bots as $bot) {
                if ($bot->pid && $bot->status === 'running') {
                    $isRunning = $this->checkProcessRunning($bot->pid);

                    if (!$isRunning) {
                        $bot->update([
                            'status' => 'stopped',
                            'pid' => null
                        ]);
                    }
                }
            }

            return response()->json($bots);

        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    private function checkProcessRunning($pid)
    {
        try {
            $isWindows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';

            if ($isWindows) {
                $output = shell_exec("tasklist /FI \"PID eq {$pid}\" 2>&1");
                return strpos($output, (string) $pid) !== false;
            } else {
                $output = shell_exec("ps -p {$pid} 2>&1");
                return strpos($output, (string) $pid) !== false;
            }
        } catch (\Exception $e) {
            return false;
        }
    }
}