<?php

namespace App\Http\Controllers;

use App\Models\Bot;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class FileManagerController extends Controller
{
    public function index()
    {
        $botsPath = storage_path('app/bots');
        $botFolders = [];

        if (File::exists($botsPath)) {
            $folders = File::directories($botsPath);

            foreach ($folders as $folder) {
                $folderName = basename($folder);
                $bot = Bot::where('file_path', $folder)->first();

                $files = File::allFiles($folder);
                $totalSize = 0;

                foreach ($files as $file) {
                    $totalSize += $file->getSize();
                }

                $botFolders[] = [
                    'name' => $folderName,
                    'bot_name' => $bot ? $bot->name : 'Unknown',
                    'path' => $folder,
                    'file_count' => count($files),
                    'size' => $this->formatBytes($totalSize),
                    'modified' => date('d/m/Y H:i', File::lastModified($folder))
                ];
            }
        }

        return view('filemanager', compact('botFolders'));
    }

    public function browse($folder)
    {
        $folderPath = storage_path('app/bots/' . $folder);

        if (!file_exists($folderPath)) {
            return redirect()->route('filemanager')->with('error', 'Folder tidak ditemukan!');
        }

        $files = collect(\File::allFiles($folderPath))->map(function ($file) {
            return [
                'name' => $file->getFilename(),
                'path' => $file->getRelativePath(),
                'relative_path' => $file->getRelativePathname(), // TAMBAH INI
                'size' => $this->formatBytes($file->getSize()),
                'extension' => strtoupper($file->getExtension() ?: 'file'),
                'modified' => date('d/m/Y H:i', $file->getMTime()),
                'raw_extension' => strtolower($file->getExtension())
            ];
        });

        return view('filemanager.browse', compact('folder', 'files'));
    }

    public function edit($folder, $file)
    {
        $filePath = storage_path('app/bots/' . $folder . '/' . $file);

        if (!file_exists($filePath)) {
            return redirect()->route('filemanager')->with('error', 'File tidak ditemukan!');
        }

        $content = file_get_contents($filePath);
        $fileName = basename($filePath);

        return view('filemanager.edit', compact('folder', 'file', 'content', 'fileName'));
    }

    public function update(Request $request, $folder, $file)
    {
        $filePath = storage_path('app/bots/' . $folder . '/' . $file);

        if (!file_exists($filePath)) {
            return redirect()->route('filemanager')->with('error', 'File tidak ditemukan!');
        }

        try {
            file_put_contents($filePath, $request->input('content'));
            return redirect()->route('filemanager.browse', $folder)->with('success', 'File berhasil diupdate!');
        } catch (\Exception $e) {
            return back()->with('error', 'Gagal update file: ' . $e->getMessage());
        }
    }

    public function deleteFile($folder, $file)
    {
        $filePath = storage_path('app/bots/' . $folder . '/' . $file);

        if (!file_exists($filePath)) {
            return redirect()->route('filemanager.browse', $folder)->with('error', 'File tidak ditemukan!');
        }

        try {
            unlink($filePath);
            return redirect()->route('filemanager.browse', $folder)->with('success', 'File berhasil dihapus!');
        } catch (\Exception $e) {
            return back()->with('error', 'Gagal menghapus file: ' . $e->getMessage());
        }
    }

    public function delete($folder)
    {
        $folderPath = storage_path('app/bots/' . $folder);

        if (File::exists($folderPath)) {
            // Cek apakah ada bot yang menggunakan folder ini
            $bot = Bot::where('file_path', $folderPath)->first();

            if ($bot && $bot->status === 'running') {
                return redirect()->route('filemanager')->with('error', 'Bot masih berjalan! Stop bot terlebih dahulu.');
            }

            File::deleteDirectory($folderPath);

            if ($bot) {
                $bot->delete();
            }

            return redirect()->route('filemanager')->with('success', 'Folder berhasil dihapus!');
        }

        return redirect()->route('filemanager')->with('error', 'Folder tidak ditemukan!');
    }

    private function formatBytes($bytes, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, $precision) . ' ' . $units[$i];
    }
}